% Finaquant Analytics - www.finaquant.com
% Copyright: Tunc Ali Ktkcoglu  2012, version: 6August2012
% TITLE: Efficient risk frontier
%
% Deriving minimum or maximum expected risk points (i.e. standard deviation) 
% and corresponding asset weights for a given expected returns with 
% constrained optimization function fmincon()
%
% Central download page:
% http://finaquant.com/download
clear all

% Create artificial historical return data
MarketIndex = 4 + 2 * randn(60,1);                      % mean = 4, stddev = 2
Asset1 = (MarketIndex * 0.5) + 3 + 2 * randn(60,1);     % positive correlation with market
Asset2 = (MarketIndex * -0.3) + 6 + 4 * randn(60,1);    % negative correlation with market
Asset3 = 8 + 5 * randn(60,1);                           % no correlation with market

% 3 assets --> N x 3 matrix
HistoricalReturns = [Asset1 Asset2 Asset3];

% get expected (mean) returns for each asset
global ExpectedReturns
ExpectedReturns = mean(HistoricalReturns)'

% get covariance matrix
global CovarianceMatrix 
CovarianceMatrix = cov(HistoricalReturns)

% Convert covariance matrix to correlation matrix
% see:
% Converting between correlation and covariance matrices
% http://blogs.sas.com/content/iml/2010/12/10/converting-between-correlation-and-covariance-matrices/
StdDeviations = sqrt(diag(CovarianceMatrix));           % vector of standard deviations
InvStdDeviations = inv(diag(StdDeviations));            % InvStdDev = 1 / StdDev
CorrelationMatrix = InvStdDeviations * CovarianceMatrix * InvStdDeviations

% calculate portfolio variance for given covariance matrix and asset weights
% Note: sum of all asset weights must be equal to 1
AssetWeights = [0.2 0.5 0.3]';
PortfolioVariance = PortfolioRisk(CovarianceMatrix, AssetWeights)

% calculate expected portfolio weights for given weights and expected
% return of each asset
ExpectedPortfolioReturn = PortfolioReturn(ExpectedReturns, AssetWeights)

%**************************************************************************
% FIND ASSET WEIGHTS FOR MINIMUM AND MAXIMUM PORTFOLIO RISK
%**************************************************************************
% constrained optimization problem with fmincon()

% equality constraints: The sum of all weights must be 1
% Aeq * AssetWeights = beq
Aeq = ones(1, length(AssetWeights))
beq = 1

% lower and upper bounds for weights vector AssetWeights
% lb <= w <= ub
lb = zeros(length(AssetWeights), 1)
ub = ones(length(AssetWeights), 1)

% initial weights, starting point
w0 = rand(length(AssetWeights), 1);
w0 = w0 / sum(w0);

% no additional inequality constraints
A = [];
b = [];

% objective function whose output is to be minimized: 
% @ObjFuncMinPortfolioRisk(Weights)
ObjFuncMinPortfolioRisk(w0)

% optimize weights for minimum portfolio variance for a given covariance matrix
[OptWeightsMinVar, MinPortfolioVariance] = fmincon(@ObjFuncMinPortfolioRisk,w0,A,b,Aeq,beq,lb,ub);

% optimize weights for maximum portfolio variance for a given covariance matrix
[OptWeightsMaxVar, MaxPortfolioVariance] = fmincon(@ObjFuncMaxPortfolioRisk,w0,A,b,Aeq,beq,lb,ub);
MaxPortfolioVariance = -1 * MaxPortfolioVariance;

%**************************************************************************
% EFFICIENT MIN RISK FRONTIER: Min risk for given return
% FIND MIN RISK (AND CORRESPONDING ASSET WEIGHTS) FOR A GIVEN PORTFOLIO RETURN
%**************************************************************************
% Portfolio return can ve any value between MinPortfolioReturn and
% MaxPortfolioReturn
MinPortfolioReturn = min(ExpectedReturns);
MaxPortfolioReturn = max(ExpectedReturns);

% construct efficient return frontier: Min risk for given return
N = 10;   % divide expected return axis to N points
global PortfolioReturn
ReturnInterval = (MaxPortfolioReturn - MinPortfolioReturn) / (N-1);

% matrices for storing results
MinRiskFrontierWeights = zeros(length(AssetWeights), N);
MinRiskFrontierReturns = zeros(1, N);
MinRiskFrontierRisk =  zeros(1, N);

% fmincon options
options.MaxFunEvals = 5000;
% options.Display = 'off';
options.Algorithm = 'interior-point';

for i=1:N
    i
    PortfolioReturn = MinPortfolioReturn + (i-1) * ReturnInterval;

    % linear equality constraints: 
    % 1) The sum of all weights must be 1
    % 2) expected portfolio return = PortfolioReturn
    Aeq = [ones(1, length(AssetWeights)); ExpectedReturns'];
    beq = [1 ; PortfolioReturn]
    
    % optimize weights for minimum portfolio risk subject to condition that
    % portfolio return= PortfolioReturn
    [OptWeightsMinRisk, MinPortfolioVar] = fmincon(@ObjFuncMinPortfolioRisk,w0,A,b,Aeq,beq,lb,ub,[],options);
    
    % store results
    MinRiskFrontierWeights(:, i) = OptWeightsMinRisk';
    MinRiskFrontierReturns(1, i) = PortfolioReturn;
    MinRiskFrontierRisk(1,i) = sqrt(MinPortfolioVar);
end

% plot efficient min risk frontier
figure2 = figure;
axes2 = axes('Parent',figure2,'YGrid','on','XGrid','on');
box(axes2,'on');
hold(axes2,'all');
plot(MinRiskFrontierRisk,MinRiskFrontierReturns,'LineWidth',2,'Color',[0 0 1]);
title({'Efficient min risk frontier'});
xlabel('portfolio risk (StdDev)');
ylabel('portfolio return in %');

%**************************************************************************
% EFFICIENT MAX RISK FRONTIER: Max risk for given return
% FIND MAX RISK (AND CORRESPONDING ASSET WEIGHTS) FOR A GIVEN PORTFOLIO RETURN
%**************************************************************************

% construct efficient return frontier: Min risk for given return
N = 10;   % divide expected return axis to N points
ReturnInterval = (MaxPortfolioReturn - MinPortfolioReturn) / (N-1);

% matrices for storing results
MaxRiskFrontierWeights = zeros(length(AssetWeights), N);
MaxRiskFrontierReturns = zeros(1, N);
MaxRiskFrontierRisk =  zeros(1, N);

% fmincon options
options.MaxFunEvals = 5000;
% options.Display = 'off';
options.Algorithm = 'interior-point';

for i=1:N
    i
    PortfolioReturn = MinPortfolioReturn + (i-1) * ReturnInterval;

    % linear equality constraints: 
    % 1) The sum of all weights must be 1
    % 2) expected portfolio return = PortfolioReturn
    Aeq = [ones(1, length(AssetWeights)); ExpectedReturns'];
    beq = [1 ; PortfolioReturn]
    
    % optimize weights for minimum portfolio risk subject to condition that
    % portfolio return= PortfolioReturn
    [OptWeightsMaxRisk, MaxPortfolioVar] = fmincon(@ObjFuncMaxPortfolioRisk,w0,A,b,Aeq,beq,lb,ub,[],options);
    
    % store results
    MaxRiskFrontierWeights(:, i) = OptWeightsMaxRisk';
    MaxRiskFrontierReturns(1, i) = PortfolioReturn;
    MaxRiskFrontierRisk(1,i) = sqrt(-1 * MaxPortfolioVar);
end

% plot efficient min risk frontier
figure3 = figure;
axes3 = axes('Parent',figure3,'YGrid','on','XGrid','on');
box(axes3,'on');
hold(axes3,'all');
plot(MaxRiskFrontierRisk,MaxRiskFrontierReturns,'LineWidth',2,'Color',[0 0 1]);
title({'Efficient max risk frontier'});
xlabel('portfolio risk (StdDev)');
ylabel('portfolio return in %');

% min/max efficient risk frontiers together
figure4 = figure;
axes4 = axes('Parent',figure4,'YGrid','on','XGrid','on');
box(axes4,'on');
hold(axes4,'all');

plot(MinRiskFrontierRisk,MinRiskFrontierReturns,'LineWidth',2,'Color',[1 0 0]);
plot(MaxRiskFrontierRisk,MaxRiskFrontierReturns,'LineWidth',2,'Color',[0 0 1]);

title({'Efficient min/max risk frontiers (red/blue)'});
xlabel('portfolio risk (StdDev)');
ylabel('portfolio return in %');

%**************************************************************************
% EFFICIENT RETURN FRONTIER: Max return for given risk
% FIND MAX RETURNS (AND CORRESPONDING ASSET WEIGHTS) FOR A GIVEN PORTFOLIO RISK
%**************************************************************************
% portfolio risk can be any value between MinPortfolioVariance and
% MaxPortfolioVariance
global PortfolioVar

% construct efficient return frontier: Max return for given risk
N = 10;   % divide risk (stdev) axis to N points
MinPortfolioStdDev = sqrt(MinPortfolioVariance);
MaxPortfolioStdDev = sqrt(MaxPortfolioVariance);
RiskInterval = (MaxPortfolioStdDev - MinPortfolioStdDev) / (N-1);

% matrices for storing results
FrontierWeights = zeros(length(AssetWeights), N);
FrontierReturns = zeros(1, N);
FrontierRisk =  zeros(1, N);

% fmincon options
options.MaxFunEvals = 5000;
% options.Display = 'off';
options.Algorithm = 'interior-point';

for i=1:N
    i
    PortfRisk = MinPortfolioStdDev + (i-1) * RiskInterval;
    PortfolioVar = PortfRisk ^ 2;
    % optimize weights for maximum portfolio return subject to condition that
    % portfolio risk = PortfolioVar
    [OptWeightsMaxReturn, MaxPortfolioReturn] = fmincon(@ObjFuncMaxPortfolioReturn,w0,A,b,Aeq,beq,lb,ub,@CondFuncPortfolioRisk, options);
    
    % store results
    FrontierWeights(:, i) = OptWeightsMaxReturn';
    FrontierReturns(1, i) = -1 * MaxPortfolioReturn;
    FrontierRisk(1,i) = PortfRisk;
end 

% plot efficient return frontier
figure1 = figure;
axes1 = axes('Parent',figure1,'YGrid','on','XGrid','on');
box(axes1,'on');
hold(axes1,'all');
plot(FrontierRisk,FrontierReturns,'LineWidth',2,'Color',[1 0 0]);
title({'Efficient return frontier'});
xlabel('portfolio risk (StdDev)');
ylabel('portfolio return in %');